//////////////////////////////////////////////////////////
//
//	AccountManager.java
//
package Alkindi.Services.ServicesImpl;

import oracle.jdbc.driver.*;
import java.sql.*;
import java.math.*;
import javax.transaction.*;
import javax.naming.*;
import Alkindi.Data.*;
import Alkindi.Services.*;
import Alkindi.Services.Util.*;
import Alkindi.Services.InternalData.*;
import javax.ejb.SessionContext;
import javax.rmi.PortableRemoteObject;

/* 
$Header: AccountManager.java, 54, 5/8/01 9:56:13 AM, Schwartz, Joe$
$Log: 
 54   Alkindi Development1.53        5/8/01 9:56:13 AM    Schwartz, Joe  
      Changed use of InternalEJBManager to InternalComponentManager. Removed
      dependencies on EJB from these classes. Cleaned up comments.
 53   Alkindi Development1.52        4/27/01 3:41:11 PM   Schwartz, Joe  
      Cleaned up utility object creation in createMember() and updateMember()
      methods.
 52   Alkindi Development1.51        4/26/01 2:37:14 PM   Schwartz, Joe  
      Modifed to account for new InternalData package.
 51   Alkindi Development1.50        4/26/01 1:11:02 PM   Schwartz, Joe   Moved
      into new ServicesImpl package.
 50   Alkindi Development1.49        4/26/01 11:18:29 AM  Schwartz, Joe  
      Changed to account for connection being set to autocommit; removed several
      calls to con.commit().
 49   Alkindi Development1.48        3/14/01 5:47:31 PM   Schwartz, Joe  
      Removed sessCtx member. Added abstract method getXA to get UserTransaction
      object. Made entire class abstract to force implementation of getXA.
 48   Alkindi Development1.47        2/23/01 4:33:01 PM   Schwartz, Joe  
      Changed to accomodate new RecMgr.getPredictedRating() which now returns a
      float.
 47   Alkindi Development1.46        2/13/01 5:55:28 PM   Schwartz, Joe  
      Changed to account for new Product id int type and SparseRatingsArray.
 46   Alkindi Development1.45        1/25/01 9:58:04 PM   Schwartz, Joe  
      Working on speed, sensibility.
 45   Alkindi Development1.44        1/21/01 4:43:42 PM   Schwartz, Joe   Fixed
      handling of some UserDetails.
 44   Alkindi Development1.43        1/16/01 1:04:21 PM   Schwartz, Joe   Added
      pass-through exception handler in validateUserQuestions so AlkExcept goes
      on up in case user fails challenge.
 43   Alkindi Development1.42        1/12/01 10:39:16 AM  Schwartz, Joe  
      getUserDetails was using the wrong field name for SPORTS_ID, preventing
      any sports from being returned in AccountInfo.
 42   Alkindi Development1.41        1/12/01 12:28:43 AM  Schwartz, Joe  
      Simplified transaction control.
 41   Alkindi Development1.40        12/28/00 8:08:59 PM  Schwartz, Joe  
      Changed members for which return value doesn't matter to return void.
      Added more javadoc comments.
 40   Alkindi Development1.39        12/28/00 2:42:04 PM  Schwartz, Joe  
      Standardized methods to use final parameters and fully-qualified types
      where necessary. Makes Rose happy.
 39   Alkindi Development1.38        12/28/00 1:08:25 PM  Schwartz, Joe   Added
      Version Control header info.
      Synchronized methods with IAccountManager interface.
 38   Alkindi Development1.37        12/28/00 12:05:51 PM Schwartz, Joe   
 37   Alkindi Development1.36        12/26/00 6:57:08 PM  Schwartz, Joe   
 36   Alkindi Development1.35        12/26/00 5:36:57 PM  Schwartz, Joe   
 35   Alkindi Development1.34        12/22/00 5:30:22 PM  Schwartz, Joe   
 34   Alkindi Development1.33        12/20/00 5:44:53 PM  Schwartz, Joe   
 33   Alkindi Development1.32        12/19/00 3:51:57 PM  Schwartz, Joe   
 32   Alkindi Development1.31        12/18/00 12:05:33 PM Schwartz, Joe   Moved
      internal data classes to Utils package & regenerated classes from Rose.
 31   Alkindi Development1.30        12/18/00 11:22:07 AM Schwartz, Joe   
 30   Alkindi Development1.29        12/10/00 3:29:31 PM  Schwartz, Joe  
      Corrected passing of AlkExcept back up stack in getScreenName.
 29   Alkindi Development1.28        12/6/00 7:11:57 PM   Schwartz, Joe   Added
      catch blocks for generic exceptions and finally blocks to close all
      database connections.
 28   Alkindi Development1.27        12/5/00 1:23:11 PM   Schwartz, Joe   Added
      more protection for null pointers in getScreenName & getMyList.
 27   Alkindi Development1.26        12/4/00 3:21:17 PM   Schwartz, Joe   Fixes
      for SQL exception in getUserDetailsList: illegal column name. Changed to
      use column number.
 26   Alkindi Development1.25        12/4/00 12:53:08 PM  Schwartz, Joe  
      Cleaned up -- deleted large swaths of commented-out code in
      getAccountInfo().
 25   Alkindi Development1.24        12/3/00 6:29:40 PM   Stanton, Sacha  
 24   Alkindi Development1.23        12/1/00 12:23:17 PM  Schwartz, Joe   
 23   Alkindi Development1.22        11/30/00 3:13:58 AM  Schwartz, Joe   
 22   Alkindi Development1.21        11/29/00 2:06:33 PM  Schwartz, Joe  
      Changed to use stored procs for insert/update operations.
 21   Alkindi Development1.20        11/29/00 1:31:05 PM  Schwartz, Joe   
 20   Alkindi Development1.19        11/22/00 6:50:16 PM  Schwartz, Joe  
      Modified to check for duplicate email in createMember and update Member.
 19   Alkindi Development1.18        11/20/00 10:52:33 PM Schwartz, Joe   
 18   Alkindi Development1.17        11/18/00 9:27:49 PM  Schwartz, Joe   
 17   Alkindi Development1.16        11/14/00 12:53:11 PM Schwartz, Joe   
 16   Alkindi Development1.15        11/8/00 2:59:01 PM   Schwartz, Joe   
 15   Alkindi Development1.14        11/7/00 2:04:10 PM   Schwartz, Joe  
      Modified to close DB connections.
 14   Alkindi Development1.13        11/6/00 7:05:57 PM   Schwartz, Joe   
 13   Alkindi Development1.12        11/2/00 12:31:10 PM  Schwartz, Joe  
      Chagned to use new procs for USER_PREFERRED_CATEGORY table.
 12   Alkindi Development1.11        10/31/00 7:04:45 PM  Schwartz, Joe   
 11   Alkindi Development1.10        10/22/00 10:39:35 AM Schwartz, Joe  
      Correction to AppianDelivery 10.20.00
 10   Alkindi Development1.9         10/22/00 10:38:38 AM Schwartz, Joe  
      AppianDelivery 10.20.00
 9    Alkindi Development1.8         10/22/00 10:37:53 AM Schwartz, Joe  
      AppianDelivery 10.20.00
 8    Alkindi Development1.7         10/20/00 1:11:25 PM  Schwartz, Joe   
 7    Alkindi Development1.6         10/20/00 11:03:18 AM Schwartz, Joe  
      Changed to correctly user result sets and break out of switch statements
      in getAccountInfo().
 6    Alkindi Development1.5         10/17/00 2:22:56 PM  Schwartz, Joe  
      Delivery 10.16.00
 5    Alkindi Development1.4         10/7/00 4:34:29 PM   Schwartz, Joe   
 4    Alkindi Development1.3         10/7/00 4:30:55 PM   Schwartz, Joe   
 3    Alkindi Development1.2         10/7/00 4:25:55 PM   Schwartz, Joe   Files
      as delivered to Alkindi Oct. 5, 2000.
 2    Alkindi Development1.1         10/6/00 4:17:13 PM   Schwartz, Joe   
 1    Alkindi Development1.0         9/26/00 4:13:27 PM   Schwartz, Joe   
$
$NoKeywords$
 */

/**
 * Implementation of IAccountManager.
 */
public abstract class AccountManager implements IAccountManager 
{
	protected final String cName = "AccountManager";
	protected LogManager logger = null;
	
	/**
	 * Adds a given product to the specified User's MyList.
	 * @param user The User whose MyList will get the new Product.
	 * @param product The Product to add.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A4B66070196
	 */
	public void addToMyList(final Alkindi.Data.SystemUser user, final Alkindi.Data.Product product) throws AlkExcept 
	{
		final String mName = "addToMyList";
		String errmsg = "";
		Connection con = null;

		try {
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{call pkg_ALKINDI_USER.sp_INS_USER_MY_LIST(?,?,?)}");
			cs.setLong(1, user.id);
			cs.setLong(2, product.id);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(3);
			cs.close();
			switch(err) {
				case 0:
				break;
				case 6515:
					errmsg = 	"addToMyList: user not found in USER_MY_LIST table.";
					logger.err(cName, mName, errmsg);
					throw new AlkExcept(errmsg, 6515);

				case 6516:
					errmsg = "general error in addToMyList.";
					logger.err(cName, mName, errmsg);
					throw new AlkExcept(errmsg, 6516);
			}


		}
		catch(AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			errmsg = "SQL Exception: "+ se.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 6004);
		}
		catch(Exception e) {
			errmsg = "Unexpected error: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2421);
		}
		finally {
			try {
				con.close();
			}
			catch (Exception e) {
			}
		}
	}
	
	/**
	 * Attempts to make the specified SystemUser an Alkindi Member using the given AccountInfo.<p>
	 * <b><i>Exception Codes</i></b>&nbsp;This method will throw an AlkExcept exceptionto indicate that member creation failed. The reason for failure will be  contained in the AlkExcept string as well as its code. The AlkExcept codes are as follows:<br>
	 * <b>6014</b>&nbsp; -&nbsp;The requested ScreenName contains an obscenity.
	 * <b>6015</b>&nbsp;-&nbsp;The requested ScreenName is already taken.
	 * <b>6016</b>&nbsp;-&nbsp;The requested Email address is already taken.
	 * 
	 * @param user The User who will be turned into a Member
	 * @param acctInfo The AccountInfo to use in creating the Member
	 * @return boolean
	 * @throws AlkExcept
	 * @roseuid 3A4B660700BB
	 */
	public void createMember(final Alkindi.Data.SystemUser user, Alkindi.Data.AccountInfo acctInfo) throws AlkExcept 
	{

		final String mName = "createMember";
		String errmsg = "";
		Connection con = null;

		logger.dbgLog(cName, mName, "Starting.");
		UserTransaction xa = getXA();
		try {
			xa.begin();

			if(isScreenNameObscene(acctInfo.screenName)) {
				errmsg = "ScreenName contains obscenity: " + acctInfo.screenName;
				logger.err(cName, mName, errmsg);
				throw new AlkExcept(errmsg, 6014);
			}

			con = AlkConn.getConnection();

			CallableStatement cs = con.prepareCall("{ call PKG_ALKINDI_USER.SP_INS_USER_DETAIL(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?) }");

			cs.setLong		(1,user.id);
			cs.setString	(2,acctInfo.email);
			cs.setString	(3,acctInfo.screenName);
			cs.setString	(4,acctInfo.password);
			cs.setInt		(5,acctInfo.question1);
			cs.setString	(6,acctInfo.answer1);
			cs.setInt		(7,acctInfo.question2);
			cs.setString	(8,acctInfo.answer2);
			cs.setString	(9,acctInfo.postalCode);
			cs.setInt		(10,acctInfo.sex);
			cs.setDate		(11,acctInfo.birthDate);
			cs.setInt		(12,acctInfo.education);
			cs.setLong		(13,acctInfo.weeklyEntExp);
			//cs.setInt		(14,2);  //Status of Member = 2;
			cs.setInt		(14,acctInfo.advisorStatus);
			cs.setString	(15, acctInfo.wantsEmail ? "Y" : "N");
			cs.setString	(16, acctInfo.wantsNews ? "Y" : "N");
			cs.setString	(17, acctInfo.wantsPromo ? "Y" : "N");
			cs.registerOutParameter(18, Types.INTEGER);

			cs.execute();
			int err = cs.getInt(18);
			cs.close();
			logger.dbgLog(cName, mName, "Finished insert into user_detail.");
			if (err != 0)
				throw new AlkExcept("Error in sp_INS_USER_DETAIL: ", err);

			AcctMgrDBUtils UD = new AcctMgrDBUtils();
			if (acctInfo.activityList.size() > 0) {        // This is for Activity.
				UD.updateUserActivity(con,user,acctInfo.activityList);
			}
			if (acctInfo.genreList.size() > 0) {        // This is for Genre Prefs.
				UD.updateUserGenre(con,user,acctInfo.genreList);
			}

			if (acctInfo.interestList.size() > 0) {        // This is for Interest.
				UD.updateUserInterest(con,user,acctInfo.interestList);
			}
			if (acctInfo.languageList.size() > 0) {        // This is for Language.
				UD.updateUserLanguage(con,user,acctInfo.languageList);
			}

			if (acctInfo.professionList.size() > 0) {        // This is for Profession.
				UD.updateUserProfession(con,user,acctInfo.professionList);
			}
			if (acctInfo.sportList.size() > 0) {        // This is for Sports.
				UD.updateUserSport(con,user,acctInfo.sportList);
			}
			logger.dbgLog(cName, mName, "Finished other inserts.");

			//	Commit transaction
			//
			xa.commit();
		} //END try

		catch (AlkExcept ae) {
			try { xa.rollback(); }	catch(Exception e) {}
			throw ae;
		}
		catch (SQLException se) {
			try { xa.rollback(); }	catch(Exception e) {}
			errmsg = "SQL Exception: " + se.getMessage();
			logger.err(cName, mName, errmsg);
            throw new AlkExcept(errmsg + " in " + mName, 6004);
        }
		catch(NullPointerException ne) {
			try { xa.rollback(); }	catch(Exception e) {}
			ne.printStackTrace();
			errmsg =  "NullPointerException: " + ne.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2412);
		}
		catch(Exception e) {
			try { xa.rollback(); }	catch(Exception e2) {}
			errmsg = "Unexpected error: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2411);
		}
		finally {
			try { 	con.close(); } catch (Exception e){}

		}
	}
	
	/**
	 * Creates a new SystemUser object. The user can rate products (and theoretically obtain recommendations), but cannot log in to the system. 
	 * @return SystemUser
	 * @throws AlkExcept
	 * @roseuid 3A4B660602DE
	 */
	public Alkindi.Data.SystemUser createUser() throws AlkExcept 
	{
		final String mName = "createUser";

		SystemUser su = new SystemUser(0);
		String errmsg = "";
		Connection con = null;

		try {
			con = AlkConn.getConnection();

			CallableStatement cs  = con.prepareCall(
			"{ call pkg_ALKINDI_USER.sp_CREATE_USER_ID(?,?)}");

			cs.setLong(1, 1);
			cs.registerOutParameter(1, Types.BIGINT);
			cs.registerOutParameter(2, Types.INTEGER);
			cs.execute();

            long userid = cs.getLong(1);
            int  error  = cs.getInt(2);

            cs.close();

			switch (error) {

			case 0 :
                su.id = userid;
                break;

            case 6500 :
				errmsg = "Error in sp_CREATE_USER_ID: User Could Not Be Created";
				logger.err(cName, mName, errmsg);
                throw new AlkExcept(errmsg, 6500);

			case 6501 :
				errmsg = "Error in sp_CREATE_USER_ID: Other Errors for Creating User";
				logger.err(cName, mName, errmsg);
                throw new AlkExcept(errmsg, 6501);
            }
        }    //END try
		catch(AlkExcept ae) {
			throw ae;
		}

		catch (SQLException se) {
			errmsg = "SQL Exception: "+ se.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 6004);
        }
		catch(Exception e) {
			errmsg = "Unexpected error " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2403);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
        return su;
	}
	
	/**
	 * Returns the AccountInfo for the given SystemUser.
	 * @param user The SystemUser whose AccountInfo is to be returned.
	 * @return AccountInfo
	 * @throws AlkExcept
	 * @roseuid 3A4B660700EA
	 */
	public Alkindi.Data.AccountInfo getAccountInfo(final Alkindi.Data.SystemUser user) throws AlkExcept 
	{
		final String mName = "getAccountInfo";
		String errmsg = "";
		Connection con = null;

		AccountInfo o_ai = new AccountInfo();

		try{
		//USER_DETAILS
		//
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall(
				"{call pkg_ALKINDI_USER.sp_SEL_USER_DETAIL(?,?,?)}");

			cs.setLong (1,user.id);
			cs.registerOutParameter(2,OracleTypes.CURSOR);
			cs.registerOutParameter(3,Types.INTEGER);
			cs.execute();

			ResultSet rs = ((OracleCallableStatement)cs).getCursor(2);
			rs.next();
			int error = cs.getInt(3);

			switch (error){
			case 0:
				o_ai.email		= rs.getString("EMAIL");
				o_ai.screenName		= rs.getString("SCREENNAME");
				o_ai.password		= rs.getString("PASSWORD");
				o_ai.question1		= rs.getInt("QUESTION1");
				o_ai.answer1		= rs.getString("ANSWER1");
				o_ai.question2		= rs.getInt("QUESTION2");
				o_ai.answer2		= rs.getString("ANSWER2");
				o_ai.postalCode		= rs.getString("HOME_ZIP_CODE");
				o_ai.sex			= rs.getInt("SEX_ID");
				o_ai.birthDate		= rs.getDate("BIRTHDATE");
				o_ai.education		= rs.getInt("EDUCATION_ID");
				o_ai.weeklyEntExp	= rs.getLong("WEEKLY_ENT_EXP");
				o_ai.userStatus		= rs.getInt("USER_STATUS_ID");
				o_ai.advisorStatus	= rs.getInt("ADVISOR_STATUS_ID");
				o_ai.wantsEmail		= rs.getString("WANTS_EMAIL").startsWith("Y");
				o_ai.wantsNews 		= rs.getString("WANTS_NEWS").startsWith("Y");
				o_ai.wantsPromo		= rs.getString("WANTS_PROMO").startsWith("Y");
				logger.dbgLog(cName, mName, "Wants email: " + rs.getString("WANTS_EMAIL") + " Wants News: " + rs.getString("WANTS_NEWS"));
				break;

			case 6545:
				errmsg = "Error in sp_SEL_USER_DETAIL: Unable to Select User Details";
				logger.err(cName, mName, errmsg);
				throw new AlkExcept(errmsg,6545);
			}

			rs.close();
			cs.close();

			AcctMgrDBUtils UD = new AcctMgrDBUtils();

			//ACTIVITY
			//
			o_ai.activityList = UD.getUserDetailList(con, user, "{call pkg_ALKINDI_USER.sp_SEL_USER_ACTIVITY(?,?,?)}", "ACTIVITY_ID", cName, mName);

			// GENRE
			//
			o_ai.genreList = UD.getUserDetailList(con, user, "{call pkg_ALKINDI_USER.sp_SEL_USER_PREFERRED_CATEGORY(?,?,?)}", "PRODUCT_CATEGORY_ID", cName, mName);

			// INTEREST
			//
			o_ai.interestList = UD.getUserDetailList(con, user, "{call pkg_ALKINDI_USER.sp_SEL_USER_INTEREST(?,?,?)}", "INTEREST_ID", cName, mName);

			// LANGUAGE
			//
			o_ai.languageList = UD.getUserDetailList(con, user, "{call pkg_ALKINDI_USER.sp_SEL_USER_LANGUAGE(?,?,?)}", "LANGUAGE_ID", cName, mName);

			// PROFESSION
			//
			o_ai.professionList = UD.getUserDetailList(con, user, "{call pkg_ALKINDI_USER.sp_SEL_USER_PROFESSION(?,?,?)}", "PROFESSION_ID", cName, mName);

			// SPORTS
			//
			o_ai.sportList = UD.getUserDetailList(con, user, "{call pkg_ALKINDI_USER.sp_SEL_USER_SPORTS(?,?,?)}", "SPORTS_ID", cName, mName);

		}
		catch (SQLException se) {
			errmsg = "SQL Exception: " + se.getMessage();
			logger.err(cName, mName, errmsg);
            throw new AlkExcept(errmsg + " in " + mName, 6004);
        }
		catch(NullPointerException ne) {
			ne.printStackTrace();
			errmsg =  "NullPointerException: " + ne.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2413);
		}
		catch(AlkExcept ae) {
			errmsg = "AlkExcept: " + ae.toString();
			logger.err(cName, mName, errmsg);
			throw (AlkExcept)ae.fillInStackTrace();
		}
		catch(Exception e) {
			errmsg = "Unexpected error: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2414);
		}
		finally {
			try {
				con.close();
			}
			catch (Exception e){
			}
		}

		return o_ai;
	}
	
	/**
	 * Returns the given User's MyList
	 * @param user The SystemUser in question.
	 * @return MyList
	 * @throws AlkExcept
	 * @roseuid 3A4B66070213
	 */
	public Alkindi.Data.MyList getMyList(final Alkindi.Data.SystemUser user) throws AlkExcept 
	{
		final String mName = "getMyList";
		String errmsg = "";
		MyList ml = new MyList();
		Connection con = null;
		IRecommendationManager recMgr = null;
		try {
			//	Get a reference to a RecommendationManager EJB.
			//
			recMgr = InternalComponentManager.getRecMgr();

			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{call pkg_ALKINDI_USER.sp_SEL_USER_MY_LIST(?,?,?)}");
			cs.setLong(1, user.id);
			cs.registerOutParameter(2, OracleTypes.CURSOR);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(3);
			switch (err){
				case 0:
				ResultSet rs = ((OracleCallableStatement)cs).getCursor(2);

				while (rs.next()) {
					int pid = rs.getInt("PRODUCT_ID");
					java.sql.Date dateAdded = rs.getDate("DATE_ADDED");
					if (rs == null || dateAdded == null) {
						logger.err(cName, mName, "Bad Mylist item returned for user id=" + user.id);
						continue;
					}
					//	Get predicted rating for product, carefully.
					//
					float predRating = 0f;
					try {
						predRating = recMgr.getPredictedRating(user, new Product(pid));
					}catch (AlkExcept ae) {
						logger.err(cName, mName, ae.toString());
					}
					Recommendation rec = new Recommendation(user, new Product(pid), predRating, 0);
					ml.add(new MyListItem(rec, dateAdded));
				}
				rs.close();
				break;

				case 6203:
					errmsg = "Database error in getMyList.";
					logger.err(cName, mName, errmsg);
					throw new AlkExcept(errmsg, 6203);
			}

			cs.close();
		}
		catch(AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			errmsg = "SQL Exception: "+ se.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 6004);
		}
		catch(Exception e) {
			errmsg = "Unexpected error: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2418);
		}
		finally {
			try {
				con.close();
				InternalComponentManager.remove(recMgr);
			}
			catch (Exception e) {
			}
		}
		return ml;
	}
	
	/**
	 * Retrives the Validation Question IDs associated with the given SystemUser
	 * @param user The SystemUser for whom Validation Questions are to be looked up.
	 * @return int[]
	 * @throws AlkExcept
	 * @roseuid 3A4B660603D8
	 */
	public int[] getQuestions(final java.lang.String login) throws AlkExcept 
	{

		final String mName = "getQuestions";
		String errmsg = "";

		int[] question = new int[2];

        question[0] = 0;
        question[1] = 0;

		Connection con = null;

        try {
			con = AlkConn.getConnection();
		    CallableStatement cs  = con.prepareCall("{ call pkg_ALKINDI_USER.sp_SELECT_QUESTION_ID(?,?,?,?)}");

            cs.setString(1, login);
            cs.registerOutParameter(2, Types.INTEGER);
            cs.registerOutParameter(3, Types.INTEGER);
            cs.registerOutParameter(4, Types.INTEGER);
            cs.execute();

            int error = cs.getInt(4);

            switch (error) {

            case 0 :
                question[0] = cs.getInt(2);
                question[1] = cs.getInt(3);
                break;

            case 6506 :
				con.close();
				errmsg = "Error in sp_UPDATE_USER_STATUS_ID: No Data Found During Select question_id";
				logger.err(cName, mName, errmsg);
                throw new AlkExcept(errmsg + " in " + mName, 6506);
            }
			cs.close();
        }    //END try
		catch(AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			errmsg = "SQL Exception: " + se.getMessage();
			logger.err(cName, mName, errmsg);
            throw new AlkExcept(errmsg + " in " + mName, 6004);
        }
		catch(Exception e) {
			errmsg = "Unexpected error: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2407);
		}
		finally {
			try {
				con.close();
			}
			catch (Exception e){
			}
		}

        return question;
	}
	
	/**
	 * Retrieves the given SystemUser's Screen Name.
	 * @param user The SystemUser for whom the Screen Name is to be lookup up.
	 * @return String
	 * @throws AlkExcept
	 * @roseuid 3A4B66070242
	 */
	public java.lang.String getScreenName(final Alkindi.Data.SystemUser user) throws AlkExcept 
	{
		final String mName = "getScreenName";
		String errmsg = "";
		Connection con = null;

		try {
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_USER.sp_SELECT_SCREENNAME(?, ?, ?) }");
			cs.setLong(1, user.id);
			cs.registerOutParameter(2, Types.CHAR);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.execute();
			int rc = cs.getInt(3);
			if (rc != 0) {
				errmsg = "Error in sp_SELECT_SCREENNAME";
				logger.err(cName, mName, errmsg);
				throw new AlkExcept(errmsg, rc);
			}
			String screenName = cs.getString(2);
			cs.close();
			if (screenName == null)
				screenName = "";
			return screenName;
		}
		catch(AlkExcept ae) {
			throw (AlkExcept)ae.fillInStackTrace();
		}
		catch (SQLException se) {
			se.printStackTrace();
			errmsg = "SQL Exception: " + se.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 6004);
		}
		catch(Exception e) {
			errmsg = "Unexpected error";
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2402);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * Initializesthe log publisher.
	 * @return void
	 * @roseuid 3A4B660602CE
	 */
	public void initLogging() 
	{
		if (logger == null)
			logger = new LogManager();
	}
	
	/**
	 * Checks if the given Screen Name is obscene. <i>Not implemented yet.<i>
	 * @param name The Screen Name to check.
	 * @return boolean
	 * @roseuid 3A4B66070119
	 */
	protected boolean isScreenNameObscene(final java.lang.String name) 
	{

        boolean tf = false;
        return tf;
	}
	
	/**
	 * Removes the given Product from a User's MyList.
	 * @param user The SystemUser whose MyList will be changed.
	 * @param product The Product to remove.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A4B660701D4
	 */
	public void removeFromMyList(final Alkindi.Data.SystemUser user, final Alkindi.Data.Product product) throws AlkExcept 
	{
		final String mName = "removeFromMyList";
		String errmsg = "";
		Connection con = null;

		try {
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{call pkg_ALKINDI_USER.sp_DEL_USER_MY_LIST(?,?,?)}");
			cs.setLong(1, user.id);
			cs.setLong(2, product.id);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(3);
			if (err != 0)
				throw new AlkExcept("Error in removeFromMyList", err);
			cs.close();
		}
		catch(AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			errmsg = "SQL Exception: "+ se.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 6004);
		}
		catch(Exception e) {
			errmsg = "Unexpected error: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2422);
		}
		finally {
			try { con.close(); } catch (Exception e) {}
		}
	}
	
	/**
	 * Updates the given member's AccountInfo. <br>
	 * <b><i>Exception Codes</i></b>&nbsp;If the routine is unable to update the AccountInfo, an AlkExcept is thrown with a String message and error code expressing the reason for failure. The codes are as follows:<br>
	 * <b>6014</b>&nbsp; -&nbsp;The requested ScreenName contains an obscenity.
	 * <b>6015</b>&nbsp;-&nbsp;The requested ScreenName is already taken.
	 * <b>6016</b>&nbsp;-&nbsp;The requested Email address is already taken.
	 * @param user The SystemUser whose information is to updated.
	 * @param acctInfo The updates AccountInfo.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A4B6607007D
	 */
	public void updateMember(final Alkindi.Data.SystemUser user, Alkindi.Data.AccountInfo acctInfo) throws AlkExcept 
	{
		final String mName = "updateMember";
		String errmsg = "";

		Connection con = null;
		UserTransaction xa = getXA();

		if(isScreenNameObscene(acctInfo.screenName)) {
			errmsg = "ScreenName contains obscenity: " + acctInfo.screenName;
			logger.log(cName, mName, errmsg);
			throw new AlkExcept(errmsg, 6014);
		}

		//	Print out AccountInfo fields
		//
		/*logger.dbgLog(cName, mName, "AcctInfo bday = " + acctInfo.birthDate);
		logger.dbgLog(cName, mName, "Preparing update: ");
		java.lang.reflect.Field flds[] = acctInfo.getClass().getFields();
		for (int idx=0; idx<flds.length; idx++) {
			try {
				logger.dbgLog(cName, mName, "\n" + flds[idx].getName() + " = ");
				logger.dbgLog(cName, mName, flds[idx].get(acctInfo).toString() );
			}
			// Ignore nulls fields for now...
			//
			catch (NullPointerException npe) {
			}
			catch (IllegalAccessException iae) {
			}
		}*/

		try {
			xa.begin();
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call PKG_ALKINDI_USER.SP_UPD_USER_DETAIL (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?) }");
			cs.setLong		(1, user.id);
			cs.setString	(2,acctInfo.email);
			cs.setString	(3,acctInfo.screenName);
			cs.setString	(4,acctInfo.password);
			cs.setInt		(5,acctInfo.question1);
			cs.setString	(6,acctInfo.answer1);
			cs.setInt		(7,acctInfo.question2);
			cs.setString	(8,acctInfo.answer2);
			cs.setString	(9,acctInfo.postalCode);
			cs.setInt		(10,acctInfo.sex);
			cs.setDate		(11,acctInfo.birthDate);
			cs.setInt		(12,acctInfo.education);
			cs.setLong		(13,acctInfo.weeklyEntExp);
			cs.setInt		(14,acctInfo.advisorStatus);
			cs.setString	(15, acctInfo.wantsEmail ? "Y" : "N");
			cs.setString	(16, acctInfo.wantsNews ? "Y" : "N");
			cs.setString	(17, acctInfo.wantsPromo ? "Y" : "N");
			cs.registerOutParameter(18, Types.INTEGER);

			cs.executeUpdate();
			int err = cs.getInt(18);
			if (err != 0 )
				throw new AlkExcept("Error in PKG_ALKINDI_USER.SP_UPD_USER_DETAIL", err);
			cs.close();
			//con.close();

			//	Update Activities
			//
			AcctMgrDBUtils UD = new AcctMgrDBUtils();
			UD.updateUserActivity(con,user,acctInfo.activityList);

			// 	Update Genres
			//
			UD.updateUserGenre(con,user,acctInfo.genreList);

			//	Update Interests
			//
			UD.updateUserInterest(con,user,acctInfo.interestList);

			//	Update Languages
			//
			UD.updateUserLanguage(con,user,acctInfo.languageList);

			//	Update Profession
			//
			logger.dbgLog(cName, mName, "profession something ELSE list size=" + acctInfo.professionList.size());
			UD.updateUserProfession(con,user,acctInfo.professionList);

			//	Update Sports
			//
			UD.updateUserSport(con,user,acctInfo.sportList);
			
			//	Commit transaction
			//
			xa.commit();

		} //END try
		catch(AlkExcept ae) {
			try { xa.rollback(); } catch (Exception e){ }
			logger.err(cName, mName, ae.getMessage());
			throw ae;
		}
		catch (SQLException se) {
			try { xa.rollback(); } catch (Exception e){ }
			errmsg = "SQL Exception: " + se.getMessage();
			logger.err(cName, mName, errmsg);
            throw new AlkExcept(errmsg + " in " + mName, 6004);
        }
		catch(NullPointerException ne) {
			try { xa.rollback(); } catch (Exception e){ }
			errmsg = "NullPointerException: " + ne.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2412);
		}
		catch(Exception e) {
			try { xa.rollback(); } catch (Exception e2){ }		
			errmsg = "Unexpected error: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2410);
		}
		finally {
			try { con.close(); } catch (Exception e){ }
		}
	}
	
	/**
	 * Updates the given User's password.
	 * @param user The SystemUser whose password is to be updated.
	 * @param oldPassword The User's old password.
	 * @param newPassword The User's new password.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A4B6607002E
	 */
	public void updatePassword(final Alkindi.Data.SystemUser user, final java.lang.String oldPassword, final java.lang.String newPassword) throws AlkExcept 
	{
		final String mName = "updatePassword";
		String errmsg = "";

		Connection con = null;

        try {
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_USER.sp_UPDATE_PASSWORD(?,?,?,?)}");

			cs.setLong   (1, user.id);
			cs.setString (2, oldPassword);
			cs.setString (3, newPassword);
			cs.registerOutParameter(4, Types.INTEGER);
			cs.execute();
			int error = cs.getInt(4);
			cs.close();

			switch (error){

			case  0:
				break;

			case 6511:
				errmsg = "Error in sp_UPDATE_PASSWORD: Unable to Update Password";
				logger.err(cName, mName, errmsg);
				throw new AlkExcept(errmsg,6511);

			case 6512:
				errmsg = "Error in sp_UPDATE_PASSWORD: Other Errors Found During Password Update";
				logger.err(cName, mName, errmsg);
				throw new AlkExcept(errmsg,6512);

			}
		} //END try
		catch (SQLException se) {
			errmsg = "SQL Exception: " + se.getMessage();
			logger.err(cName, mName, errmsg);
            throw new AlkExcept(errmsg + " in " + mName, 6004);
        }
		catch(Exception e) {
			errmsg = "Unexpected error: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2409);
		}
		finally {
			try { con.close(); } catch (Exception e){ }
		}
	}
	
	/**
	 * Changes the given User's status.
	 * @param user The SystemUser whose status is to be changed.
	 * @param newStatus The User's new status.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A4B66060399
	 */
	public void updateUserStatus(final Alkindi.Data.SystemUser user, final int newStatus) throws AlkExcept 
	{
		final String mName = "updateUserStatus";
		String errmsg = "";

		Connection con = null;
        try {
			con = AlkConn.getConnection();
            CallableStatement cs  = con.prepareCall(
                "{ call pkg_ALKINDI_USER.sp_UPDATE_USER_STATUS_ID(?,?,?)}");

            cs.setLong(1, user.id);
            cs.setInt(2, newStatus);
            cs.registerOutParameter(3, Types.INTEGER);
            cs.execute();

            int error = cs.getInt(3);

            cs.close();

            switch (error) {

		        case 0 :
                break;

            case 6509 :
				errmsg = "Error in sp_UPDATE_USER_STATUS_ID: User_id Not Found During Update";
				logger.err(cName, mName, errmsg);
                throw new AlkExcept(errmsg + " in " + mName, 6509);

            case 6510 :
				errmsg = "Error in sp_UPDATE_USER_STATUS_ID: Other Errors Found During Update";
				logger.err(cName, mName, errmsg);
                throw new AlkExcept(errmsg + " in " + mName, 6510);
            }
        }    //END try

		catch (SQLException se) {
			errmsg = "SQL Exception: " + se.getMessage();
			logger.err(cName, mName, errmsg);
            throw new AlkExcept(errmsg + " in " + mName, 6004);
        }
		catch(Exception e) {
			errmsg = "Unexpected error: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2406);
		}
		finally {
			try { con.close();} catch (Exception e){}
		}
	}
	
	/**
	 * Attempts to validate the given login and password against the database. If successful, the SystemUser corresponding to the given login is returned.
	 * @param login
	 * @param password
	 * @return SystemUser
	 * @throws AlkExcept
	 * @roseuid 3A4B660602DF
	 */
	public Alkindi.Data.SystemUser validateUser(final java.lang.String login, final java.lang.String password) throws AlkExcept 
	{
		final String mName = "validateUser";
		String errmsg = "";

		SystemUser o_su = new SystemUser(0);
		Connection con = null;

        try {
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
                "{ call pkg_ALKINDI_USER.sp_VERIFY_LOGIN_PASSWORD(?,?,?,?)}");

            cs.setString(1, login);
            cs.setString(2, password);
            cs.registerOutParameter(3, Types.INTEGER);
            cs.registerOutParameter(4, Types.INTEGER);
            cs.execute();

            long userid = cs.getLong(3);
            int  error  = cs.getInt(4);

            cs.close();

            switch (error) {

            case 0 :
                o_su.id = userid;
                break;

            case 6502 :
				errmsg = "Error in sp_VERIFY_LOGIN_PASSWORD: Account Suspended";
				logger.err(cName, mName, errmsg);
                throw new AlkExcept(errmsg, 6502);

            case 6503 :
				errmsg = "Error in sp_VERIFY_LOGIN_PASSWORD: No Data Found About the User";
				logger.err(cName, mName, errmsg);
                throw new AlkExcept(errmsg, 6503);
            }
        } //END try

		catch (SQLException se) {
			errmsg = "SQL Exception: " + se.getMessage();
			logger.err(cName, mName, errmsg);
            throw new AlkExcept(errmsg + " in " + mName, 6004);
        }
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (Exception e) {
			errmsg = "Unexpected error " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2404);
		}
		finally {
			try {
				con.close();
			}
			catch(Exception e) {
			}
		}
        return o_su;
	}
	
	/**
	 * Attempts to validate the given login, questions and answers. If successful, it returns the SystemUser corresponding to the login.
	 * @param login 
	 * @param question1 The ID of the first validation question.
	 * @param answer1 The first validation answer.
	 * @param question2 The ID of the second validation question.
	 * @param answer2 The second validation answer.
	 * @roseuid 3A4B6606031C
	 */
	public Alkindi.Data.SystemUser validateUserQuestions(final java.lang.String login, final int question1, final java.lang.String answer1, final int question2, final java.lang.String answer2) throws AlkExcept 
	{
		final String mName = "validateUserQuestions";
		String errmsg = "";
		Connection con = null;
        SystemUser o_su = null;

        try {
			con = AlkConn.getConnection();
            CallableStatement cs  = con.prepareCall(
                "{ call pkg_ALKINDI_USER.sp_VERIFY_QUESTION_ANSWER(?,?,?,?,?,?,?)}");

            cs.setString(1, login);
            cs.setInt(2, question1);
            cs.setInt(3, question2);
            cs.setString(4, answer1);
            cs.setString(5, answer2);
            cs.registerOutParameter(6, Types.INTEGER);
            cs.registerOutParameter(7, Types.INTEGER);
            cs.execute();

            int userid = cs.getInt(6);
            int error  = cs.getInt(7);

            cs.close();

            switch (error) {

            case 0 :
				o_su = new SystemUser(userid);
				break;

			case 6507:
				throw new AlkExcept("", 6507);

			default:
				errmsg = "Unknown error in pkg_ALKINDI_USER.sp_VERIFY_QUESTION_ANSWER";
				logger.err(cName, mName, errmsg);
				throw new AlkExcept(errmsg, error);

   }
        } //END try
		catch(AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			errmsg = "SQL Exception: " + se.getMessage();
			logger.err(cName, mName, errmsg);
            throw new AlkExcept(errmsg + " in " + mName,6004);
        }
		catch(Exception e) {
			errmsg = "Unexpected error: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg + " in " + mName, 2405);
		}
		finally {
			try { con.close(); } catch (Exception e){ }
		}
        return o_su;
	}
	
	/**
	 * Retrieves a UserTransaction object to manage database transactions.
	 * @returns UserTransaction
	 * @roseuid 3AAFE7AC0148
	 */
	public abstract javax.transaction.UserTransaction getXA();
}
